<?php
// Habilitar exibição de erros para debug
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include_once 'dbconfig.php';
include_once 'includes/antisql.php';

// Verificar se a conexão foi estabelecida
if (!$conn) {
    echo json_encode(['success' => false, 'message' => 'Erro de conexão com banco de dados: ' . mysqli_connect_error()]);
    exit;
}

// Processar ações via POST - DEVE estar no início antes de qualquer saída HTML
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    $acao = isset($_POST['acao']) ? $_POST['acao'] : '';
    $id_pedido = isset($_POST['id_pedido']) ? (int)$_POST['id_pedido'] : 0;
    $id_cliente = isset($_POST['id_cliente']) ? (int)$_POST['id_cliente'] : 0;
    $id_vendas = isset($_POST['id_vendas']) ? (int)$_POST['id_vendas'] : 0;
    
    if ($acao === 'inserir') {
        $produto = isset($_POST['produto']) ? $_POST['produto'] : '';
        $valor = isset($_POST['valor']) ? str_replace(',', '.', $_POST['valor']) : 0;
        $data = isset($_POST['data']) ? $_POST['data'] : date('Y-m-d');
        
        if (!empty($produto) && $valor > 0) {
            $sql = "INSERT INTO vendas (id_pedido, produto, valor, data2) VALUES (?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            
            if ($stmt) {
                $stmt->bind_param("isds", $id_pedido, $produto, $valor, $data);
                
                if ($stmt->execute()) {
                    echo json_encode(['success' => true, 'message' => 'Venda adicionada com sucesso!']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erro ao adicionar venda: ' . $stmt->error]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
            }
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Produto e valor são obrigatórios.']);
            exit;
        }
    } elseif ($acao === 'alterar' && $id_vendas > 0) {
        $produto = isset($_POST['produto']) ? $_POST['produto'] : '';
        $valor = isset($_POST['valor']) ? str_replace(',', '.', $_POST['valor']) : 0;
        $data = isset($_POST['data']) ? $_POST['data'] : date('Y-m-d');
        
        if (!empty($produto) && $valor > 0) {
            $sql = "UPDATE vendas SET produto = ?, valor = ?, data2 = ? WHERE id = ?";
            $stmt = $conn->prepare($sql);
            
            if ($stmt) {
                $stmt->bind_param("sdsi", $produto, $valor, $data, $id_vendas);
                
                if ($stmt->execute()) {
                    echo json_encode(['success' => true, 'message' => 'Venda atualizada com sucesso!']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erro ao atualizar venda: ' . $stmt->error]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
            }
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Produto e valor são obrigatórios.']);
            exit;
        }
    } elseif ($acao === 'excluir' && $id_vendas > 0) {
        $sql = "DELETE FROM vendas WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_vendas);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Venda excluída com sucesso!']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao excluir venda: ' . $stmt->error]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
        }
        exit;
    }
}

// Processar ação buscar_venda via GET - Precisa estar no início para evitar saída HTML
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['acao']) && $_GET['acao'] === 'buscar_venda') {
    header('Content-Type: application/json');
    
    $id_vendas = isset($_GET['id_vendas']) ? (int)$_GET['id_vendas'] : 0;
    
    if ($id_vendas > 0) {
        $sql = "SELECT * FROM vendas WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_vendas);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                $venda = $result->fetch_assoc();
                
                if ($venda) {
                    // Formatar valor para exibição
                    $valor_formatado = number_format($venda['valor'], 2, ',', '.');
                    
                    echo json_encode([
                        'success' => true,
                        'data' => [
                            'produto' => $venda['produto'],
                            'valor_formatado' => $valor_formatado,
                            'data_venda' => $venda['data2']
                        ]
                    ]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Venda não encontrada.']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao buscar venda.']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID da venda inválido.']);
    }
    exit;
}

// Receber parâmetros
$id_pedido = isset($_GET['id_pedido']) ? (int)$_GET['id_pedido'] : 0;
$id_cliente = isset($_GET['id_cliente']) ? (int)$_GET['id_cliente'] : 0;
$id_vendas = isset($_GET['id_vendas']) ? (int)$_GET['id_vendas'] : 0;
$acao = isset($_GET['acao']) ? $_GET['acao'] : '';

// Se for uma requisição para listar vendas (AJAX), retornar apenas a tabela
if ($acao === 'listar_vendas' && $_SERVER['REQUEST_METHOD'] === 'GET') {
    // Buscar informações do pedido e cliente
    $info_pedido = null;
    if ($id_pedido > 0 && $id_cliente > 0) {
        $sql = "SELECT p.id as pedido_id, p.data, u.nome as cliente_nome 
                FROM pedidos p 
                LEFT JOIN usuarios u ON p.id_cliente = u.id 
                WHERE p.id = ? AND p.id_cliente = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("ii", $id_pedido, $id_cliente);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                $info_pedido = $result->fetch_assoc();
            }
        }
    }
    
    // Buscar vendas do pedido
    $vendas = [];
    if ($id_pedido > 0) {
        $sql = "SELECT * FROM vendas WHERE id_pedido = ? ORDER BY data2 DESC, id DESC";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_pedido);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                while ($row = $result->fetch_assoc()) {
                    $vendas[] = $row;
                }
            }
        }
    }
    
    // Calcular total de vendas
    $total_vendas = 0;
    foreach ($vendas as $venda) {
        $total_vendas += $venda['valor'];
    }
    
    // Retornar apenas a tabela HTML
    ?>
    <!-- Lista de Vendas -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Lista de Vendas</h5>
                    <span class="badge bg-primary fs-6">Total: R$ <?php echo number_format($total_vendas, 2, ',', '.'); ?></span>
                </div>
                <div class="card-body">
                    <!-- Filtros -->
                    <div class="row mb-3">
                        <div class="col-md-3">
                            <input type="text" class="form-control" id="filtroNumero" placeholder="Filtrar por ID...">
                        </div>
                        <div class="col-md-3">
                            <input type="text" class="form-control" id="filtroProduto" placeholder="Filtrar por produto...">
                        </div>
                        <div class="col-md-3">
                            <input type="text" class="form-control" id="filtroValor" placeholder="Filtrar por valor...">
                        </div>
                        <div class="col-md-3">
                            <input type="date" class="form-control" id="filtroData" placeholder="Filtrar por data...">
                        </div>
                    </div>
                    
                    <?php if (count($vendas) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="tabelaVendas">
                            <thead class="table-dark">
                                <tr>
                                    <th>ID</th>
                                    <th>Produto</th>
                                    <th>Valor</th>
                                    <th>Data</th>
                                    <th>Ações</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($vendas as $venda): ?>
                                <tr>
                                    <td><?php echo $venda['id']; ?></td>
                                    <td><?php echo htmlspecialchars($venda['produto']); ?></td>
                                    <td>R$ <?php echo number_format($venda['valor'], 2, ',', '.'); ?></td>
                                    <td><?php 
                                         // Verificar e formatar data da venda corretamente
                                         if (!empty($venda['data2']) && $venda['data2'] !== '0000-00-00' && $venda['data2'] !== '1970-01-01') {
                                             $data_original = $venda['data2'];
                                             
                                             // Se a data já está no formato dd/mm/yyyy, apenas exibir
                                             if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $data_original)) {
                                                 echo $data_original;
                                             }
                                             // Se está no formato yyyy-mm-dd, converter para dd/mm/yyyy
                                             else if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data_original)) {
                                                 $timestamp = strtotime($data_original);
                                                 if ($timestamp && $timestamp > 0) {
                                                     echo date('d/m/Y', $timestamp);
                                                 } else {
                                                     echo 'Data inválida';
                                                 }
                                             }
                                             // Outros formatos
                                             else {
                                                 echo $data_original;
                                             }
                                         } else {
                                             echo 'Sem data';
                                         }
                                     ?></td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-warning me-1" 
                                                onclick="editarVenda(<?php echo $venda['id']; ?>)" title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-danger" 
                                                onclick="excluirVenda(<?php echo $venda['id']; ?>)" title="Excluir">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php else: ?>
                    <div class="alert alert-info text-center">
                        <i class="fas fa-info-circle"></i> Nenhuma venda encontrada para este pedido.
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    <?php
    exit;
}

// Receber parâmetros
$id_pedido = isset($_GET['id_pedido']) ? (int)$_GET['id_pedido'] : 0;
$id_cliente = isset($_GET['id_cliente']) ? (int)$_GET['id_cliente'] : 0;
$id_vendas = isset($_GET['id_vendas']) ? (int)$_GET['id_vendas'] : 0;
$acao = isset($_GET['acao']) ? $_GET['acao'] : '';

// Agora que todo o processamento AJAX foi feito, podemos iniciar o HTML
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vendas - Modal</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- FontAwesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <!-- SweetAlert2 -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
</head>
<body>

<?php

// Buscar dados para edição
$venda_edicao = null;
if ($acao === 'editar' && $id_vendas > 0) {
    $sql = "SELECT * FROM vendas WHERE id = ?";
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("i", $id_vendas);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $venda_edicao = $result->fetch_assoc();
        }
    }
}

// Buscar informações do pedido e cliente
$info_pedido = null;
if ($id_pedido > 0 && $id_cliente > 0) {
    $sql = "SELECT p.id as pedido_id, p.data, u.nome as cliente_nome 
            FROM pedidos p 
            LEFT JOIN usuarios u ON p.id_cliente = u.id 
            WHERE p.id = ? AND p.id_cliente = ?";
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("ii", $id_pedido, $id_cliente);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $info_pedido = $result->fetch_assoc();
        }
    }
}

// Buscar vendas do pedido
$vendas = [];
if ($id_pedido > 0) {
    $sql = "SELECT * FROM vendas WHERE id_pedido = ? ORDER BY data2 DESC, id DESC";
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("i", $id_pedido);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $vendas[] = $row;
            }
        }
    }
}

// Calcular total de vendas
$total_vendas = 0;
foreach ($vendas as $venda) {
    $total_vendas += $venda['valor'];
}
?>

<div class="container-fluid">
    <?php if ($info_pedido): ?>
    <div class="row mb-3">
        <div class="col-12">
            <div class="alert alert-info">
                <strong>Pedido ID:</strong> <?php echo htmlspecialchars($info_pedido['pedido_id']); ?> | 
                <strong>Data:</strong> <?php 
                    // Verificar e formatar data do pedido corretamente
                    if (!empty($info_pedido['data']) && $info_pedido['data'] !== '0000-00-00' && $info_pedido['data'] !== '1970-01-01') {
                        $data_original = $info_pedido['data'];
                        
                        // Se a data já está no formato dd/mm/yyyy, apenas exibir
                        if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $data_original)) {
                            echo $data_original;
                        }
                        // Se está no formato yyyy-mm-dd, converter para dd/mm/yyyy
                        else if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data_original)) {
                            $timestamp = strtotime($data_original);
                            if ($timestamp && $timestamp > 0) {
                                echo date('d/m/Y', $timestamp);
                            } else {
                                echo 'Data inválida';
                            }
                        }
                        // Outros formatos
                        else {
                            echo $data_original;
                        }
                    } else {
                        echo 'Sem data';
                    }
                ?> | 
                <strong>Cliente:</strong> <?php echo htmlspecialchars($info_pedido['cliente_nome']); ?>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Espaço para mensagens de alerta -->
    <div class="row mb-4">
        <div class="col-12">
            <div id="alertMessages"></div>
        </div>
    </div>
    
    <!-- Lista de Vendas -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Vendas do Pedido</h5>
                    <div class="d-flex align-items-center gap-2">
                        <button type="button" class="btn btn-success btn-sm" onclick="adicionarNovaLinha()" title="Adicionar Nova Venda">
                            <i class="fas fa-plus"></i> Nova Venda
                        </button>
                        <span class="badge bg-primary fs-6">Total: R$ <?php echo number_format($total_vendas, 2, ',', '.'); ?></span>
                    </div>
                </div>
                <div class="card-body">
                    <?php if (empty($vendas)): ?>
                    <div class="text-center text-muted py-4">
                        <i class="fas fa-shopping-cart fa-3x mb-3"></i>
                        <p>Nenhuma venda encontrada para este pedido.</p>
                    </div>
                    <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="tabelaVendas">
                            <thead class="table-dark">
                                <tr>
                                    <th>Nº</th>
                                    <th>Produto</th>
                                    <th>Valor</th>
                                    <th>Data</th>
                                    <th width="120">Ações</th>
                                </tr>
                            </thead>
                            <tfoot>
                                <tr>
                                    <td><input type="text" class="form-control form-control-sm" id="filtroNumero" placeholder="Filtrar por Nº"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filtroProduto" placeholder="Filtrar por Produto"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filtroValor" placeholder="Filtrar por Valor"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filtroData" placeholder="Filtrar por Data"></td>
                                    <td><button type="button" class="btn btn-secondary btn-sm w-100" onclick="limparFiltros()">Limpar</button></td>
                                </tr>
                            </tfoot>
                            <tbody>
                                <?php foreach ($vendas as $venda): ?>
                                <tr data-id="<?php echo $venda['id']; ?>">
                                    <td><?php echo $venda['id']; ?></td>
                                    <td class="editable-cell" data-field="produto" data-original="<?php echo htmlspecialchars($venda['produto']); ?>">
                                        <span class="cell-display"><?php echo htmlspecialchars($venda['produto']); ?></span>
                                        <input type="text" class="cell-input form-control form-control-sm" value="<?php echo htmlspecialchars($venda['produto']); ?>" style="display: none;">
                                    </td>
                                    <td class="editable-cell" data-field="valor" data-original="<?php echo $venda['valor']; ?>">
                                        <span class="cell-display">R$ <?php echo number_format($venda['valor'], 2, ',', '.'); ?></span>
                                        <input type="text" class="cell-input form-control form-control-sm valor-input" value="<?php echo number_format($venda['valor'], 2, ',', '.'); ?>" style="display: none;">
                                    </td>
                                    <td class="editable-cell" data-field="data2" data-original="<?php echo $venda['data2']; ?>">
                                        <span class="cell-display"><?php 
                                            // Verificar se a data existe e não está vazia
                                            if (!empty($venda['data2']) && $venda['data2'] !== '0000-00-00' && $venda['data2'] !== '1970-01-01') {
                                                // Tentar diferentes formatos de data
                                                $data_formatada = '';
                                                
                                                // Se já está no formato brasileiro (dd/mm/yyyy)
                                                if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $venda['data2'])) {
                                                    $data_formatada = $venda['data2'];
                                                }
                                                // Se está no formato americano (yyyy-mm-dd)
                                                else if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $venda['data2'])) {
                                                    $timestamp = strtotime($venda['data2']);
                                                    if ($timestamp && $timestamp > 0) {
                                                        $data_formatada = date('d/m/Y', $timestamp);
                                                    }
                                                }
                                                // Tentar outros formatos
                                                else {
                                                    $timestamp = strtotime($venda['data2']);
                                                    if ($timestamp && $timestamp > 0) {
                                                        $data_formatada = date('d/m/Y', $timestamp);
                                                    }
                                                }
                                                
                                                echo $data_formatada ? $data_formatada : $venda['data2'];
                                            } else {
                                                echo 'Sem data';
                                            }
                                        ?></span>
                                        <input type="date" class="cell-input form-control form-control-sm" value="<?php 
                                            // Converter data para formato yyyy-mm-dd para o input date
                                            if (!empty($venda['data2']) && $venda['data2'] !== '0000-00-00' && $venda['data2'] !== '1970-01-01') {
                                                // Se já está no formato yyyy-mm-dd
                                                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $venda['data2'])) {
                                                    echo $venda['data2'];
                                                }
                                                // Se está no formato dd/mm/yyyy, converter
                                                else if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $venda['data2'])) {
                                                    $partes = explode('/', $venda['data2']);
                                                    echo $partes[2] . '-' . $partes[1] . '-' . $partes[0];
                                                }
                                                // Tentar converter outros formatos
                                                else {
                                                    $timestamp = strtotime($venda['data2']);
                                                    if ($timestamp && $timestamp > 0) {
                                                        echo date('Y-m-d', $timestamp);
                                                    }
                                                }
                                            }
                                        ?>" style="display: none;">
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-success btn-save" onclick="salvarEdicaoInline(<?php echo $venda['id']; ?>)" title="Salvar" style="display: none;">
                                            <i class="fas fa-check"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-secondary btn-cancel" onclick="cancelarEdicaoInline(<?php echo $venda['id']; ?>)" title="Cancelar" style="display: none;">
                                            <i class="fas fa-times"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-warning btn-edit" 
                                                onclick="editarInline(<?php echo $venda['id']; ?>)" title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-danger" 
                                                onclick="excluirVenda(<?php echo $venda['id']; ?>)" title="Excluir">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.modal-xl {
    max-width: 95%;
}

.table th, .table td {
    vertical-align: middle;
    padding: 8px;
}

.btn-sm {
    padding: 4px 8px;
    font-size: 12px;
}

.form-control-sm {
    height: calc(1.5em + 0.5rem + 2px);
    padding: 0.25rem 0.5rem;
    font-size: 0.875rem;
}

.table-responsive {
    border: 1px solid #dee2e6;
    border-radius: 0.375rem;
}

.alert {
    margin-bottom: 1rem;
}

.text-center {
    text-align: center;
}

.text-muted {
    color: #6c757d;
}

.no-results {
    display: none;
    text-align: center;
    padding: 20px;
    color: #6c757d;
    font-style: italic;
}

/* Estilos para filtros no tfoot */
#tabelaVendas tfoot td {
    padding: 5px;
    background-color: #f8f9fa;
    border-top: 2px solid #dee2e6;
}

#tabelaVendas tfoot input {
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 12px;
    width: 100%;
}

#tabelaVendas tfoot input:focus {
    border-color: #80bdff;
    box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
}

#tabelaVendas tfoot button {
    font-size: 11px;
    padding: 2px 6px;
}

/* Estilos para edição inline */
.editable-cell {
    cursor: pointer;
    position: relative;
}

.editable-cell:hover {
    background-color: #f8f9fa;
}

.editable-cell.editing {
    background-color: #fff3cd;
    cursor: default;
}

.cell-input {
    width: 100%;
    border: 2px solid #007bff;
    background-color: #fff;
}

.cell-input:focus {
    border-color: #0056b3;
    box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
}

.valor-input {
    text-align: right;
}

.btn-save, .btn-cancel {
    margin-right: 2px;
}

.editing-row {
    background-color: #fff3cd !important;
}
</style>

<script>
// Submissão do formulário via AJAX
$('#formVenda').on('submit', function(e) {
    e.preventDefault();
    
    // Converter valor formatado para formato do banco
    var valorCampo = $('#valor').val();
    var valorParaBanco = valorCampo.replace(/\./g, '').replace(',', '.');
    
    // Verificar se está em modo de edição
    var idVendas = $('#btnSalvar').data('id-vendas');
    var acao = idVendas ? 'alterar' : 'inserir';
    
    // Preparar dados para envio
    var dadosEnvio = {
        acao: acao,
        produto: $('#produto').val(),
        valor: valorParaBanco,
        data: $('#data_venda').val(),
        id_pedido: <?php echo $id_pedido; ?>,
        id_cliente: <?php echo $id_cliente; ?>
    };
    
    // Adicionar ID da venda se for edição
    if (idVendas) {
        dadosEnvio.id_vendas = idVendas;
    }
    
    $.ajax({
        url: 'vendas_modal.php',
        type: 'POST',
        data: dadosEnvio,
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                // Mostrar mensagem de sucesso
                Swal.fire({
                    icon: 'success',
                    title: 'Sucesso!',
                    text: response.message,
                    timer: 1500,
                    showConfirmButton: false
                });
                
                // Atualizar tabela sem recarregar a página
                atualizarTabelaVendas();
                
                // Limpar formulário e voltar para modo inserção
                cancelarEdicao();
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Erro!',
                    text: response.message
                });
            }
        },
        error: function() {
            Swal.fire({
                icon: 'error',
                title: 'Erro!',
                text: 'Erro ao processar solicitação.'
            });
        }
    });
});

// Função para editar venda
function editarVenda(idVendas) {
    // Buscar dados da venda via AJAX
    $.ajax({
        url: 'vendas_modal.php',
        type: 'GET',
        data: {
            acao: 'buscar_venda',
            id_vendas: idVendas,
            id_pedido: <?php echo $id_pedido; ?>,
            id_cliente: <?php echo $id_cliente; ?>
        },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                // Preencher campos do formulário
                $('#produto').val(response.data.produto);
                $('#valor').val(response.data.valor_formatado);
                $('#data_venda').val(response.data.data_venda);
                
                // Alterar botão para modo edição
                $('#btnSalvar').text('Atualizar Venda').data('id-vendas', idVendas);
                $('#btnCancelar').show();
                
                // Focar no campo produto
                $('#produto').focus();
                
                // Aplicar máscara no valor
                aplicarMascaraValor();
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Erro!',
                    text: response.message || 'Erro ao carregar dados da venda.'
                });
            }
        },
        error: function(xhr, status, error) {
            var errorMessage = 'Erro ao carregar dados da venda.';
            if (xhr.responseText) {
                try {
                    var errorResponse = JSON.parse(xhr.responseText);
                    errorMessage = errorResponse.message || errorMessage;
                } catch(e) {
                    errorMessage += ' Detalhes: ' + xhr.responseText.substring(0, 100);
                }
            }
            
            Swal.fire({
                icon: 'error',
                title: 'Erro!',
                text: errorMessage
            });
        }
    });
}

// Função para cancelar edição
function cancelarEdicao() {
    // Limpar campos do formulário
    $('#produto').val('');
    $('#valor').val('');
    $('#data_venda').val('');
    
    // Voltar botão para modo inserção
    $('#btnSalvar').text('Salvar Venda').removeData('id-vendas');
    $('#btnCancelar').hide();
    
    // Focar no campo produto
    $('#produto').focus();
}

// Função para excluir venda
function excluirVenda(idVendas) {
    Swal.fire({
        title: 'Tem certeza?',
        text: 'Esta ação não pode ser desfeita!',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sim, excluir!',
        cancelButtonText: 'Cancelar'
    }).then((result) => {
        if (result.isConfirmed) {
            $.ajax({
                url: 'vendas_modal.php',
                type: 'POST',
                data: {
                    acao: 'excluir',
                    id_vendas: idVendas,
                    id_pedido: <?php echo $id_pedido; ?>,
                    id_cliente: <?php echo $id_cliente; ?>
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Excluído!',
                            text: response.message,
                            timer: 1500,
                            showConfirmButton: false
                        });
                        
                        // Atualizar tabela sem recarregar a página
                        atualizarTabelaVendas();
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Erro!',
                            text: response.message
                        });
                    }
                },
                error: function() {
                    Swal.fire({
                        icon: 'error',
                        title: 'Erro!',
                        text: 'Erro ao excluir venda.'
                    });
                }
            });
        }
    });
}

// Aplicar máscara para valor monetário usando implementação customizada
function aplicarMascaraValor() {
    const campoValor = $('#valor');
    if (campoValor.length > 0) {
        const valorAtual = campoValor.val();
        
        // Remove qualquer máscara existente
        campoValor.unmask();
        
        // Função para formatar valor em tempo real
        function formatarValor(valor) {
            // Remove tudo que não é número
            valor = valor.replace(/\D/g, '');
            
            // Se vazio, retorna vazio
            if (!valor) return '';
            
            // Converte para centavos
            valor = parseInt(valor);
            
            // Formata como moeda brasileira
            return (valor / 100).toLocaleString('pt-BR', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
        }
        
        // Evento de input para formatação em tempo real
        campoValor.off('input.mascara').on('input.mascara', function(e) {
            const cursorPos = this.selectionStart;
            const valorAnterior = $(this).val();
            const valorFormatado = formatarValor($(this).val());
            
            $(this).val(valorFormatado);
            
            // Ajustar posição do cursor
            const novaPosicao = Math.min(cursorPos + (valorFormatado.length - valorAnterior.length), valorFormatado.length);
            this.setSelectionRange(novaPosicao, novaPosicao);
        });
        
        // Evento de foco para seleção
        campoValor.off('focus.mascara').on('focus.mascara', function() {
            setTimeout(() => {
                this.select();
            }, 10);
        });
        
        // Permitir apenas números e teclas de controle
        campoValor.off('keydown.mascara').on('keydown.mascara', function(e) {
            const permitidas = [
                8, 9, 27, 13, 46, // Backspace, Tab, Esc, Enter, Delete
                35, 36, 37, 38, 39, 40, // Home, End, setas
                96, 97, 98, 99, 100, 101, 102, 103, 104, 105, // Numpad 0-9
                48, 49, 50, 51, 52, 53, 54, 55, 56, 57 // 0-9
            ];
            
            // Permitir Ctrl+A, Ctrl+C, Ctrl+V, Ctrl+X
            if ((e.ctrlKey || e.metaKey) && [65, 67, 86, 88].includes(e.keyCode)) {
                return true;
            }
            
            // Bloquear se não estiver na lista de permitidas
            if (!permitidas.includes(e.keyCode)) {
                e.preventDefault();
                return false;
            }
        });
        
        // Restaurar valor se existir
        if (valorAtual && valorAtual.trim() !== '') {
            // Se o valor já está formatado, manter
            if (valorAtual.includes(',')) {
                campoValor.val(valorAtual);
            } else {
                // Se é um número puro, formatar
                const valorNumerico = parseFloat(valorAtual.replace(',', '.'));
                if (!isNaN(valorNumerico)) {
                    const valorFormatado = (valorNumerico).toLocaleString('pt-BR', {
                        minimumFractionDigits: 2,
                        maximumFractionDigits: 2
                    });
                    campoValor.val(valorFormatado);
                }
            }
        }
    }
}

// Função para filtrar tabela de vendas
function filtrarTabela() {
    var filtroNumero = $('#filtroNumero').val().toLowerCase();
    var filtroProduto = $('#filtroProduto').val().toLowerCase();
    var filtroValor = $('#filtroValor').val().toLowerCase();
    var filtroData = $('#filtroData').val().toLowerCase();
    
    var linhasVisiveis = 0;
    
    $('#tabelaVendas tbody tr').each(function() {
        var linha = $(this);
        
        // Pular linha de "Sem Resultados" se existir
        if (linha.hasClass('no-results')) {
            return;
        }
        
        var numero = linha.find('td:eq(0)').text().toLowerCase();
        var produto = linha.find('td:eq(1)').text().toLowerCase();
        var valor = linha.find('td:eq(2)').text().toLowerCase();
        var data = linha.find('td:eq(3)').text().toLowerCase();
        
        var mostrarLinha = true;
        
        // Verificar cada filtro
        if (filtroNumero && numero.indexOf(filtroNumero) === -1) {
            mostrarLinha = false;
        }
        if (filtroProduto && produto.indexOf(filtroProduto) === -1) {
            mostrarLinha = false;
        }
        if (filtroValor && valor.indexOf(filtroValor) === -1) {
            mostrarLinha = false;
        }
        if (filtroData && data.indexOf(filtroData) === -1) {
            mostrarLinha = false;
        }
        
        if (mostrarLinha) {
            linha.show();
            linhasVisiveis++;
        } else {
            linha.hide();
        }
    });
    
    // Mostrar/ocultar mensagem "Sem Resultados"
    if (linhasVisiveis === 0) {
        if ($('#tabelaVendas .no-results').length === 0) {
            $('#tabelaVendas tbody').append('<tr class="no-results"><td colspan="5" class="text-center text-muted py-4"><i class="fas fa-search"></i><br>Nenhum resultado encontrado</td></tr>');
        }
        $('#tabelaVendas .no-results').show();
    } else {
        $('#tabelaVendas .no-results').hide();
    }
}

// Função para limpar todos os filtros
function limparFiltros() {
    $('#filtroNumero, #filtroProduto, #filtroValor, #filtroData').val('');
    $('#tabelaVendas tbody tr').show();
    $('#tabelaVendas .no-results').hide();
    
    // Focar no primeiro campo de filtro
    $('#filtroNumero').focus();
}

// Função para atualizar tabela de vendas sem recarregar a página
function atualizarTabelaVendas() {
    $.ajax({
        url: 'vendas_modal.php',
        type: 'GET',
        data: {
            id_pedido: <?php echo $id_pedido; ?>,
            id_cliente: <?php echo $id_cliente; ?>,
            acao: 'listar_vendas'
        },
        success: function(response) {
            // Encontrar o container da lista de vendas e substituir
            var containerVendas = $('.card:has(#tabelaVendas)').parent();
            if (containerVendas.length === 0) {
                // Se não encontrar pela tabela, procurar pelo título "Lista de Vendas"
                containerVendas = $('.card-header:contains("Lista de Vendas")').closest('.card').parent();
            }
            
            if (containerVendas.length > 0) {
                containerVendas.html(response);
                
                // Reaplicar eventos de filtro
                $('#filtroNumero, #filtroProduto, #filtroValor, #filtroData').off('input keyup').on('input keyup', function() {
                    filtrarTabela();
                });
                
                // Reaplicar evento de escape para limpar filtros
                $(document).off('keydown.filtros').on('keydown.filtros', function(e) {
                    if (e.key === 'Escape') {
                        limparFiltros();
                    }
                });
            } else {
                console.log('Container da tabela não encontrado, recarregando página...');
                location.reload();
            }
        },
        error: function() {
            console.log('Erro ao atualizar tabela, recarregando página...');
            location.reload();
        }
    });
}

// Aplicar máscara quando o documento estiver pronto
$(document).ready(function() {
    // Aplicar máscara imediatamente
    aplicarMascaraValor();
    
    // Também aplicar quando o campo receber foco (para casos de edição)
    $(document).on('focus', '#valor', function() {
        aplicarMascaraValor();
    });
    
    // Adicionar eventos de filtro em tempo real
    $('#filtroNumero, #filtroProduto, #filtroValor, #filtroData').on('input keyup', function() {
        filtrarTabela();
    });
    
    // Limpar filtros com tecla Escape
    $(document).on('keydown', function(e) {
        if (e.key === 'Escape') {
            limparFiltros();
        }
    });
    
    // Focar no campo produto ao carregar
    setTimeout(function() {
        $('#produto').focus();
    }, 100);
});

// Funções para edição inline
function editarInline(idVenda) {
    var linha = $('tr[data-id="' + idVenda + '"]');
    
    // Verificar se já está em modo de edição
    if (linha.hasClass('editing-row')) {
        return;
    }
    
    // Cancelar outras edições em andamento
    cancelarTodasEdicoes();
    
    // Marcar linha como em edição
    linha.addClass('editing-row');
    
    // Para cada célula editável
    linha.find('.editable-cell').each(function() {
        var celula = $(this);
        var display = celula.find('.cell-display');
        var input = celula.find('.cell-input');
        
        // Esconder display e mostrar input
        display.hide();
        input.show().focus();
        
        // Marcar célula como em edição
        celula.addClass('editing');
    });
    
    // Mostrar botões de salvar/cancelar e esconder botão editar
    linha.find('.btn-edit').hide();
    linha.find('.btn-save, .btn-cancel').show();
    
    // Aplicar máscara no campo valor
    var valorInput = linha.find('.valor-input');
    if (valorInput.length > 0) {
        aplicarMascaraValorInline(valorInput);
    }
}

function cancelarEdicaoInline(idVenda) {
    var linha = $('tr[data-id="' + idVenda + '"]');
    
    // Restaurar valores originais
    linha.find('.editable-cell').each(function() {
        var celula = $(this);
        var display = celula.find('.cell-display');
        var input = celula.find('.cell-input');
        var valorOriginal = celula.data('original');
        
        // Restaurar valor original no input
        if (celula.data('field') === 'valor') {
            var valorFormatado = parseFloat(valorOriginal).toLocaleString('pt-BR', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
            input.val(valorFormatado);
        } else if (celula.data('field') === 'produto') {
            input.val(valorOriginal);
        } else if (celula.data('field') === 'data2') {
            input.val(valorOriginal);
        }
        
        // Mostrar display e esconder input
        input.hide();
        display.show();
        
        // Remover classe de edição
        celula.removeClass('editing');
    });
    
    // Remover classe de linha em edição
    linha.removeClass('editing-row');
    
    // Mostrar botão editar e esconder botões salvar/cancelar
    linha.find('.btn-save, .btn-cancel').hide();
    linha.find('.btn-edit').show();
}

function salvarEdicaoInline(idVenda) {
    var linha = $('tr[data-id="' + idVenda + '"]');
    
    // Coletar dados dos inputs
    var produto = linha.find('[data-field="produto"] .cell-input').val();
    var valorFormatado = linha.find('[data-field="valor"] .cell-input').val();
    var data = linha.find('[data-field="data2"] .cell-input').val();
    
    // Converter valor para formato do banco
    var valorParaBanco = valorFormatado.replace(/\./g, '').replace(',', '.');
    
    // Validar dados
    if (!produto.trim()) {
        Swal.fire({
            icon: 'error',
            title: 'Erro!',
            text: 'O produto é obrigatório.'
        });
        return;
    }
    
    if (!valorParaBanco || isNaN(parseFloat(valorParaBanco))) {
        Swal.fire({
            icon: 'error',
            title: 'Erro!',
            text: 'O valor deve ser um número válido.'
        });
        return;
    }
    
    // Debug: Log dos dados que serão enviados
    console.log('Dados a serem enviados:', {
        acao: 'alterar',
        id_vendas: idVenda,
        produto: produto,
        valor: valorParaBanco,
        data: data,
        id_pedido: <?php echo $id_pedido; ?>,
        id_cliente: <?php echo $id_cliente; ?>
    });
    
    // Teste usando JavaScript puro (fetch)
    console.log('Iniciando teste de comunicação AJAX...');
    console.log('Dados que serão enviados:', {
        acao: 'alterar',
        id_vendas: idVenda,
        produto: produto,
        valor: valorParaBanco,
        data: data,
        id_pedido: <?php echo $id_pedido; ?>,
        id_cliente: <?php echo $id_cliente; ?>
    });
    
    // Criar FormData para garantir POST
    var formData = new FormData();
    formData.append('acao', 'alterar');
    formData.append('id_vendas', idVenda);
    formData.append('produto', produto);
    formData.append('valor', valorParaBanco);
    formData.append('data', data);
    formData.append('id_pedido', <?php echo $id_pedido; ?>);
    formData.append('id_cliente', <?php echo $id_cliente; ?>);
    
    console.log('Enviando requisição AJAX via fetch...');
    
    fetch('test_vendas_ajax.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(function(response) {
        console.log('Resposta recebida:', response);
        if (response.success) {
            // Atualizar displays com novos valores
            linha.find('[data-field="produto"] .cell-display').text(produto);
            linha.find('[data-field="valor"] .cell-display').text('R$ ' + parseFloat(valorParaBanco).toLocaleString('pt-BR', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            }));
            
            // Formatar e atualizar data
            if (data && data !== '0000-00-00') {
                var dataFormatada = new Date(data + 'T00:00:00').toLocaleDateString('pt-BR');
                linha.find('[data-field="data2"] .cell-display').text(dataFormatada);
            } else {
                linha.find('[data-field="data2"] .cell-display').text('Sem data');
            }
            
            // Atualizar dados originais
            linha.find('[data-field="produto"]').data('original', produto);
            linha.find('[data-field="valor"]').data('original', valorParaBanco);
            linha.find('[data-field="data2"]').data('original', data);
            
            // Sair do modo de edição
            cancelarEdicaoInline(idVenda);
            
            // Mostrar mensagem de sucesso
            Swal.fire({
                icon: 'success',
                title: 'Sucesso!',
                text: 'Venda atualizada com sucesso!',
                timer: 1500,
                showConfirmButton: false
            });
            
            // Tabela atualizada em tempo real - sem recarregar a página
        } else {
            Swal.fire({
                icon: 'error',
                title: 'Erro!',
                text: response.message || 'Erro ao atualizar venda.'
            });
        }
    })
     .catch(function(error) {
         console.log('Erro AJAX:', error);
         
         Swal.fire({
             icon: 'error',
             title: 'Erro!',
             text: 'Erro de comunicação com o servidor. Verifique o console para mais detalhes.'
         });
     });
}

function cancelarTodasEdicoes() {
    $('.editing-row').each(function() {
        var idVenda = $(this).data('id');
        cancelarEdicaoInline(idVenda);
    });
}

function aplicarMascaraValorInline(campo) {
    if (!campo || campo.length === 0) return;
    
    // Remover eventos anteriores
    campo.off('.mascaraInline');
    
    // Aplicar máscara de valor
    campo.on('input.mascaraInline', function() {
        var valor = $(this).val();
        var cursorPos = this.selectionStart;
        var valorAnterior = valor;
        
        // Remover caracteres não numéricos
        valor = valor.replace(/[^\d]/g, '');
        
        // Converter para formato monetário
        if (valor.length === 0) {
            $(this).val('');
            return;
        }
        
        // Adicionar zeros à esquerda se necessário
        while (valor.length < 3) {
            valor = '0' + valor;
        }
        
        // Inserir vírgula decimal
        var parteInteira = valor.slice(0, -2);
        var parteDecimal = valor.slice(-2);
        
        // Adicionar pontos de milhares
        parteInteira = parteInteira.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        
        var valorFormatado = parteInteira + ',' + parteDecimal;
        $(this).val(valorFormatado);
        
        // Ajustar posição do cursor
        var novaPosicao = Math.min(cursorPos + (valorFormatado.length - valorAnterior.length), valorFormatado.length);
        this.setSelectionRange(novaPosicao, novaPosicao);
    });
    
    // Selecionar tudo ao focar
    campo.on('focus.mascaraInline', function() {
        setTimeout(() => {
            this.select();
        }, 10);
    });
}

// Função para adicionar nova linha
function adicionarNovaLinha() {
    // Cancelar edições em andamento
    cancelarTodasEdicoes();
    
    // Gerar ID temporário para nova linha
    var novoId = 'novo_' + Date.now();
    
    // Criar nova linha HTML
    var novaLinha = `
        <tr data-id="${novoId}" class="editing-row nova-linha">
            <td>Novo</td>
            <td class="editable-cell" data-field="produto" data-original="">
                <span class="cell-display" style="display: none;"></span>
                <input type="text" class="form-control form-control-sm cell-input" placeholder="Digite o produto" style="display: block;">
            </td>
            <td class="editable-cell" data-field="valor" data-original="">
                <span class="cell-display" style="display: none;"></span>
                <input type="text" class="form-control form-control-sm cell-input valor-input" placeholder="0,00" style="display: block;">
            </td>
            <td class="editable-cell" data-field="data2" data-original="">
                <span class="cell-display" style="display: none;"></span>
                <input type="date" class="form-control form-control-sm cell-input" style="display: block;">
            </td>
            <td>
                <button type="button" class="btn btn-success btn-sm btn-save" onclick="salvarNovaVenda('${novoId}')" title="Salvar">
                    <i class="fas fa-save"></i>
                </button>
                <button type="button" class="btn btn-secondary btn-sm btn-cancel ms-1" onclick="cancelarNovaVenda('${novoId}')" title="Cancelar">
                    <i class="fas fa-times"></i>
                </button>
            </td>
        </tr>
    `;
    
    // Adicionar nova linha ao final da tabela
    $('#tabelaVendas tbody').append(novaLinha);
    
    // Aplicar máscara no campo valor
    var valorInput = $(`tr[data-id="${novoId}"] .valor-input`);
    aplicarMascaraValorInline(valorInput);
    
    // Focar no primeiro campo
    $(`tr[data-id="${novoId}"] input[data-field="produto"]`).focus();
}

// Função para salvar nova venda
function salvarNovaVenda(novoId) {
    var linha = $('tr[data-id="' + novoId + '"]');
    
    // Coletar dados dos inputs
    var produto = linha.find('[data-field="produto"] .cell-input').val();
    var valorFormatado = linha.find('[data-field="valor"] .cell-input').val();
    var data = linha.find('[data-field="data2"] .cell-input').val();
    
    // Converter valor para formato do banco
    var valorParaBanco = valorFormatado.replace(/\./g, '').replace(',', '.');
    
    // Validar dados
    if (!produto.trim()) {
        Swal.fire({
            icon: 'error',
            title: 'Erro!',
            text: 'O produto é obrigatório.'
        });
        return;
    }
    
    if (!valorParaBanco || isNaN(parseFloat(valorParaBanco))) {
        Swal.fire({
            icon: 'error',
            title: 'Erro!',
            text: 'O valor deve ser um número válido.'
        });
        return;
    }
    
    if (!data) {
        Swal.fire({
            icon: 'error',
            title: 'Erro!',
            text: 'A data é obrigatória.'
        });
        return;
    }
    
    // Enviar dados via AJAX
    $.ajax({
        url: 'vendas_modal.php',
        type: 'POST',
        data: {
            acao: 'inserir',
            produto: produto,
            valor: valorParaBanco,
            data: data,
            id_pedido: <?php echo $id_pedido; ?>,
            id_cliente: <?php echo $id_cliente; ?>
        },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                // Atualizar a linha com os dados salvos
                var novoIdVenda = response.id_vendas || 'novo';
                linha.attr('data-id', novoIdVenda);
                linha.removeClass('nova-linha');
                
                // Atualizar displays com valores salvos
                linha.find('[data-field="produto"] .cell-display').text(produto);
                linha.find('[data-field="valor"] .cell-display').text('R$ ' + parseFloat(valorParaBanco).toLocaleString('pt-BR', {
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 2
                }));
                
                // Formatar e atualizar data
                if (data && data !== '0000-00-00') {
                    var dataFormatada = new Date(data + 'T00:00:00').toLocaleDateString('pt-BR');
                    linha.find('[data-field="data2"] .cell-display').text(dataFormatada);
                } else {
                    linha.find('[data-field="data2"] .cell-display').text('Sem data');
                }
                
                // Atualizar dados originais
                linha.find('[data-field="produto"]').data('original', produto);
                linha.find('[data-field="valor"]').data('original', valorParaBanco);
                linha.find('[data-field="data2"]').data('original', data);
                
                // Sair do modo de edição
                linha.find('.cell-input').hide();
                linha.find('.cell-display').show();
                linha.removeClass('editing-row');
                
                // Atualizar botões para modo normal
                linha.find('td:last').html(`
                    <button type="button" class="btn btn-primary btn-sm btn-edit" onclick="editarVendaInline(${novoIdVenda})" title="Editar">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button type="button" class="btn btn-danger btn-sm btn-delete ms-1" onclick="excluirVenda(${novoIdVenda})" title="Excluir">
                        <i class="fas fa-trash"></i>
                    </button>
                `);
                
                Swal.fire({
                    icon: 'success',
                    title: 'Sucesso!',
                    text: 'Venda adicionada com sucesso!',
                    timer: 1500,
                    showConfirmButton: false
                });
                
                // Tabela atualizada em tempo real - sem recarregar a página
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Erro!',
                    text: response.message || 'Erro ao adicionar venda.'
                });
            }
        },
        error: function() {
            Swal.fire({
                icon: 'error',
                title: 'Erro!',
                text: 'Erro de comunicação com o servidor.'
            });
        }
    });
}

// Função para cancelar nova venda
function cancelarNovaVenda(novoId) {
    var linha = $('tr[data-id="' + novoId + '"]');
    linha.remove();
}
</script>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>

</body>
</html>